/**
 * @file test_idxsection.c
 *
 * @copyright Copyright  (C)  2012 Jörg Behrens <behrens@dkrz.de>
 *                                 Moritz Hanke <hanke@dkrz.de>
 *                                 Thomas Jahns <jahns@dkrz.de>
 *
 * @author Jörg Behrens <behrens@dkrz.de>
 *         Moritz Hanke <hanke@dkrz.de>
 *         Thomas Jahns <jahns@dkrz.de>
 */
/*
 * Keywords:
 * Maintainer: Jörg Behrens <behrens@dkrz.de>
 *             Moritz Hanke <hanke@dkrz.de>
 *             Thomas Jahns <jahns@dkrz.de>
 * URL: https://redmine.dkrz.de/doc/yaxt/html/index.html
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are  permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 *
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * Neither the name of the DKRZ GmbH nor the names of its contributors
 * may be used to endorse or promote products derived from this software
 * without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
 * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <assert.h>
#include <limits.h>
#include <stdlib.h>

#include <mpi.h>

#include <yaxt.h>

#include "core/ppm_xfuncs.h"

#define VERBOSE
#include "tests.h"
#include "test_idxlist_utils.h"


static void
do_tests(Xt_idxlist idxlist, Xt_int * ref_indices, Xt_int num_indices,
         struct Xt_stripe * ref_stripes, Xt_int ref_num_stripes);
static void
check_stripes(struct Xt_stripe const * stripes, Xt_int num_stripes,
              struct Xt_stripe const * ref_stripes, Xt_int ref_num_stripes);

int main(void) {

  // init mpi

  xt_mpi_call(MPI_Init(NULL, NULL), MPI_COMM_WORLD);

  xt_initialize(MPI_COMM_WORLD);

  { // test 1D section

    Xt_idxlist idxsection;

    Xt_int start = 0;
    unsigned num_dimensions = 1;
    Xt_int global_size[1] = {10};
    Xt_int local_size [1] = {5};
    Xt_int local_start[1] = {3};

    // create index section

    idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    // testing

    Xt_int ref_indices[5] = {3,4,5,6,7};
    struct Xt_stripe ref_stripes[1] = {{3,5,1}};
    do_tests(idxsection, ref_indices, 5, ref_stripes, 1);

    // clean up

    xt_idxlist_delete(idxsection);
  }

  { // test 2D section

    Xt_idxlist idxsection;

    Xt_int start = 0;
    unsigned num_dimensions = 2;
    Xt_int global_size[2] = {5,6};
    Xt_int local_size [2] = {3,2};
    Xt_int local_start[2] = {1,2};

    // create index section

    idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    // testing

    Xt_int ref_indices[6] = {8,9,14,15,20,21};
    struct Xt_stripe ref_stripes[3] = {{8,2,1}, {14,2,1}, {20,2,1}};
    do_tests(idxsection, ref_indices, 6, ref_stripes, 3);

    // clean up

    xt_idxlist_delete(idxsection);
  }

  { // test 3D section

    Xt_idxlist idxsection;

    Xt_int start = 0;
    unsigned num_dimensions = 3;
    Xt_int global_size[3] = {4,4,4};
    Xt_int local_size[3] = {4,2,2};
    Xt_int local_start[3] = {0,1,1};

    // create index section

    idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    // testing

    Xt_int ref_indices[16] = {5,6,9,10, 21,22,25,26, 37,38,41,42, 53,54,57,58};
    struct Xt_stripe ref_stripes[8] = {{5,2,1}, {9,2,1}, {21,2,1},
                                       {25,2,1}, {37,2,1}, {41,2,1},
                                       {53,2,1}, {57,2,1}};
    do_tests(idxsection, ref_indices, 16, ref_stripes, 8);

    // clean up

    xt_idxlist_delete(idxsection);
  }
  
  { // test 3D section
    Xt_idxlist idxsection;

    Xt_int start = 0;
    unsigned num_dimensions = 3;
    Xt_int global_size[3] = {3,4,5};
    Xt_int local_size[3] = {2,2,3};
    Xt_int local_start[3] = {1,1,1};

    // create index section

    idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    // testing

    Xt_int ref_indices[12] = {26,27,28,31,32,33,46,47,48,51,52,53};
    struct Xt_stripe ref_stripes[4] = {{26,3,1}, {31,3,1}, {46,3,1}, {51,3,1}};
    do_tests(idxsection, ref_indices, 12, ref_stripes, 4);

    // clean up

    xt_idxlist_delete(idxsection);
  }

  { // test 4D section
    Xt_idxlist idxsection;

    Xt_int start = 0;
    unsigned num_dimensions = 4;
    Xt_int global_size[4] = {3,4,4,3};
    Xt_int local_size[4] = {2,3,3,2};
    Xt_int local_start[4] = {0,1,1,1};

    // create index section

    idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    // testing

    Xt_int ref_indices[36] = {16,17,19,20,22,23, 28,29,31,32,34,35, 40,41,43,44,46,47,
                              64,65,67,68,70,71, 76,77,79,80,82,83, 88,89,91,92,94,95};
    struct Xt_stripe ref_stripes[18] = {{16,2,1},{19,2,1},{22,2,1}, {28,2,1},{31,2,1},{34,2,1}, {40,2,1},{43,2,1},{46,2,1},
                                        {64,2,1},{67,2,1},{70,2,1}, {76,2,1},{79,2,1},{82,2,1}, {88,2,1},{91,2,1},{94,2,1}};
    do_tests(idxsection, ref_indices, 36, ref_stripes, 18);

    // clean up

    xt_idxlist_delete(idxsection);
  }

  { // test 2D section

    Xt_idxlist idxsection;

    Xt_int start = 0;
    unsigned num_dimensions = 2;
    Xt_int global_size[2] = {5,10};
    Xt_int local_size [2] = {3,4};
    Xt_int local_start[2] = {1,2};

    // create index section

    idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    // testing

    Xt_int ref_indices[12] = {12,13,14,15,22,23,24,25,32,33,34,35};

    check_idxlist(idxsection, ref_indices, 12);

    // clean up

    xt_idxlist_delete(idxsection);
  }

  { // 1D intersection test

    Xt_idxlist idxsection_a, idxsection_b;

    Xt_int start = 0;
    unsigned num_dimensions = 1;
    Xt_int global_size_a[1] = {10};
    Xt_int local_size_a [1] = {5};
    Xt_int local_start_a[1] = {4};

    Xt_int global_size_b[1] = {15};
    Xt_int local_size_b [1] = {6};
    Xt_int local_start_b[1] = {7};

    // create index sections

    idxsection_a = xt_idxsection_new(start, num_dimensions, global_size_a,
                                     local_size_a, local_start_a);
    idxsection_b = xt_idxsection_new(start, num_dimensions, global_size_b,
                                     local_size_b, local_start_b);

    // compute intersection

    Xt_idxlist intersection;

    intersection = xt_idxlist_get_intersection(idxsection_a, idxsection_b);

    // check intersection

    Xt_int ref_indices[2] = {7,8};
    struct Xt_stripe ref_stripes[1] = {{7,2,1}};

    do_tests(intersection, ref_indices, 2, ref_stripes, 1);

    xt_idxlist_delete(intersection);
    xt_idxlist_delete(idxsection_a);
    xt_idxlist_delete(idxsection_b);
  }

  { // 1D intersection test (with empty intersection)

    Xt_idxlist idxsection_a, idxsection_b;

    Xt_int start = 0;
    unsigned num_dimensions = 1;
    Xt_int global_size_a[1] = {10};
    Xt_int local_size_a [1] = {1};
    Xt_int local_start_a[1] = {3};

    Xt_int global_size_b[1] = {10};
    Xt_int local_size_b [1] = {5};
    Xt_int local_start_b[1] = {4};

    // create index sections

    idxsection_a = xt_idxsection_new(start, num_dimensions, global_size_a,
                                     local_size_a, local_start_a);
    idxsection_b = xt_idxsection_new(start, num_dimensions, global_size_b,
                                     local_size_b, local_start_b);

    // compute intersection

    Xt_idxlist intersection;

    intersection = xt_idxlist_get_intersection(idxsection_a, idxsection_b);

    // check intersection

    Xt_int ref_indices[1] = {0};
    struct Xt_stripe ref_stripes[1] = {{0,1,1}};

    do_tests(intersection, ref_indices, 0, ref_stripes, 0);

    xt_idxlist_delete(intersection);
    xt_idxlist_delete(idxsection_a);
    xt_idxlist_delete(idxsection_b);
  }

  { // 2D intersection test

    Xt_idxlist idxsection_a, idxsection_b;

    Xt_int start = 0;
    unsigned num_dimensions = 2;
    Xt_int global_size_a[2] = {6,6};
    Xt_int local_size_a [2] = {4,2};
    Xt_int local_start_a[2] = {1,1};

    Xt_int global_size_b[2] = {6,6};
    Xt_int local_size_b [2] = {3,3};
    Xt_int local_start_b[2] = {3,2};

    // create index sections

    idxsection_a = xt_idxsection_new(start, num_dimensions, global_size_a,
                                     local_size_a, local_start_a);
    idxsection_b = xt_idxsection_new(start, num_dimensions, global_size_b,
                                     local_size_b, local_start_b);

    // compute intersection

    Xt_idxlist intersection;

    intersection = xt_idxlist_get_intersection(idxsection_a, idxsection_b);

    // check intersection

    Xt_int ref_indices[2] = {20,26};
    struct Xt_stripe ref_stripes[2] = {{20,1,1}, {26,1,1}};

    do_tests(intersection, ref_indices, 2, ref_stripes, 2);

    xt_idxlist_delete(intersection);
    xt_idxlist_delete(idxsection_a);
    xt_idxlist_delete(idxsection_b);
  }

  { // 2D test

    Xt_idxlist idxsection;

    Xt_int start = 0;
    unsigned num_dimensions = 2;
    Xt_int global_size[2] = {4,4};
    Xt_int local_size [2] = {2,2};
    Xt_int local_start[2] = {0,2};

    idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    Xt_int indices[4], ref_indices[4] = {2,3,6,7};

    xt_idxlist_get_indices(idxsection, indices);

    for (int i = 0; i < 4; ++i)
      if (indices[i] != ref_indices[i])
        PUT_ERR("error in xt_idxlist_get_indices\n");

    xt_idxlist_delete(idxsection);
  }

  { // 2D test

    Xt_idxlist idxsection;

    Xt_int start = 1;
    unsigned num_dimensions = 2;
    Xt_int global_size[2] = {4,4};
    Xt_int local_size [2] = {2,2};
    Xt_int local_start[2] = {0,2};

    idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    Xt_int indices[4], ref_indices[4] = {3,4,7,8};

    xt_idxlist_get_indices(idxsection, indices);

    for (int i = 0; i < 4; ++i)
      if (indices[i] != ref_indices[i])
        PUT_ERR("error in xt_idxlist_get_indices\n");

    xt_idxlist_delete(idxsection);
  }

  {
    // check get_positions_of_indices

    Xt_int start = 0;
    unsigned num_dimensions = 2;
    Xt_int global_size[2] = {4,4};
    Xt_int local_size [2] = {2,2};
    Xt_int local_start[2] = {0,2};

    Xt_idxlist idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    // we have indices = {2,3,6,7}
    Xt_int selection[] = {1,2,5,6,7,8};
    int num_selection = sizeof(selection) / sizeof(*selection);

    int positions[num_selection];
    int ref_positions[] = {1*0-1, 2*0+0, 5*0-1, 6*0+2, 7*0+3, 8*0-1};

    if (xt_idxlist_get_positions_of_indices(idxsection, selection, num_selection, positions, 0) != 3)
      PUT_ERR("xt_idxlist_get_position_of_indices did not return correct num_unmatched\n");

    for (int i=0; i<num_selection; i++) {
      if (positions[i] != ref_positions[i])
        PUT_ERR("xt_idxlist_get_position_of_indices did not return correct position\n");
    }

    xt_idxlist_delete(idxsection);
  }

  {
    // check get_positions_of_indices with single_match_only = 0

    Xt_int start = 0;
    unsigned num_dimensions = 2;
    Xt_int global_size[2] = {4,4};
    Xt_int local_size [2] = {2,2};
    Xt_int local_start[2] = {0,2};

    Xt_idxlist idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    // we have indices = {2,3,6,7}
    Xt_int selection[] = {2,1,5,7,6,7,7,6,8};
    int num_selection = sizeof(selection) / sizeof(*selection);

    int positions[num_selection];
    int ref_positions[] = {2*0+0, 1*0-1, 5*0-1, 7*0+3, 6*0+2, 7*0+3, 7*0+3, 6*0+2, 8*0-1};
    int single_match_only = 0;

    if (xt_idxlist_get_positions_of_indices(idxsection, selection, num_selection, positions, single_match_only) != 3)
      PUT_ERR("xt_idxlist_get_position_of_indices did not return correct num_unmatched\n");

    for (int i=0; i<num_selection; i++) {
      int p;

      xt_idxlist_get_position_of_index(idxsection, selection[i], &p);

      if (p != ref_positions[i])
        PUT_ERR("xt_idxlist_get_position_of_index did not return correct position\n");
      if (positions[i] != ref_positions[i])
        PUT_ERR("xt_idxlist_get_positions_of_indices did not return correct position\n");
    }

    xt_idxlist_delete(idxsection);
  }

  {
    // check get_positions_of_indices with single_match_only = 1

    Xt_int start = 0;
    unsigned num_dimensions = 2;
    Xt_int global_size[2] = {4,4};
    Xt_int local_size [2] = {2,2};
    Xt_int local_start[2] = {0,2};

    Xt_idxlist idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    // we have indices = {2,3,6,7}
    Xt_int selection[] = {2,1,5,7,6,7,7,6,8};
    int num_selection = sizeof(selection) / sizeof(*selection);

    int positions[num_selection];
    int ref_positions[] = {2*0+0, 1*0-1, 5*0-1, 7*0+3, 6*0+2, 7*0-1, 7*0-1, 6*0-1, 8*0-1};
    int single_match_only = 1;

    if (xt_idxlist_get_positions_of_indices(idxsection, selection, num_selection, positions, single_match_only) != 6)
      PUT_ERR("xt_idxlist_get_position_of_indices did not return correct num_unmatched\n");

    for (int i=0; i<num_selection; i++) {
      if (positions[i] != ref_positions[i])
        PUT_ERR("xt_idxlist_get_positions_of_indices did not return correct position\n");
    }

    xt_idxlist_delete(idxsection);
  }

  {
    // check idxsection_get_intersection_with_other_idxlist

    Xt_int start = 0;
    unsigned num_dimensions = 2;
    Xt_int global_size[2] = {4,4};
    Xt_int local_size [2] = {2,2};
    Xt_int local_start[2] = {0,2};

    Xt_idxlist idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    // we have indices = {2,3,6,7}
    Xt_int sel_idx[] = {2,1,5,7,6,7,7,6,8};
    int num_sel_idx = sizeof(sel_idx) / sizeof(*sel_idx);

    Xt_idxlist sel_idxlist = xt_idxvec_new(sel_idx, num_sel_idx);

    Xt_idxlist inter_idxlist = xt_idxlist_get_intersection(idxsection, sel_idxlist);

    Xt_int ref_inter_idx[] = {2,6,6,7,7,7};
    int num_ref_inter_idx =  sizeof(ref_inter_idx) / sizeof(*ref_inter_idx);

    check_idxlist(inter_idxlist, ref_inter_idx, num_ref_inter_idx);

    xt_idxlist_delete(inter_idxlist);
    xt_idxlist_delete(sel_idxlist);
    xt_idxlist_delete(idxsection);
  }

  { // test 2D section with negative global size

    for (int i = 0; i < 16; ++i) {

      Xt_idxlist idxsection;

      Xt_int start = 0;
      unsigned num_dimensions = 2;
      Xt_int global_size[4][2] = {{5,10},{5,-10},{-5,10},{-5,-10}};
      Xt_int local_size [4][2] = {{3,4},{3,-4},{-3,4},{-3,-4}};
      Xt_int local_start[2] = {1,2};

      // create index section

      idxsection = xt_idxsection_new(start, num_dimensions, global_size[i >> 2],
                                     local_size[i & 3], local_start);

      // testing

      Xt_int ref_indices[16][12] =
        {{12, 13, 14, 15, 22, 23, 24, 25, 32, 33, 34, 35},
         {15, 14, 13, 12, 25, 24, 23, 22, 35, 34, 33, 32},
         {32, 33, 34, 35, 22, 23, 24, 25, 12, 13, 14, 15},
         {35, 34, 33, 32, 25, 24, 23, 22, 15, 14, 13, 12},
         {17, 16, 15, 14, 27, 26, 25, 24, 37, 36, 35, 34},
         {14, 15, 16, 17, 24, 25, 26, 27, 34, 35, 36, 37},
         {37, 36, 35, 34, 27, 26, 25, 24, 17, 16, 15, 14},
         {34, 35, 36, 37, 24, 25, 26, 27, 14, 15, 16, 17},
         {32, 33, 34, 35, 22, 23, 24, 25, 12, 13, 14, 15},
         {35, 34, 33, 32, 25, 24, 23, 22, 15, 14, 13, 12},
         {12, 13, 14, 15, 22, 23, 24, 25, 32, 33, 34, 35},
         {15, 14, 13, 12, 25, 24, 23, 22, 35, 34, 33, 32},
         {37, 36, 35, 34, 27, 26, 25, 24, 17, 16, 15, 14},
         {34, 35, 36, 37, 24, 25, 26, 27, 14, 15, 16, 17},
         {17, 16, 15, 14, 27, 26, 25, 24, 37, 36, 35, 34},
         {14, 15, 16, 17, 24, 25, 26, 27, 34, 35, 36, 37}};

      check_idxlist(idxsection, ref_indices[i], 12);

      // clean up

      xt_idxlist_delete(idxsection);
    }
  }

  { // test 2D section with negative global size

    for (int i = 0; i < 16; ++i) {

      Xt_idxlist idxsection;

      Xt_int start = 0;
      unsigned num_dimensions = 2;
      Xt_int global_size[4][2] = {{5,6},{5,-6},{-5,6},{-5,-6}};
      Xt_int local_size [4][2] = {{2,3},{2,-3},{-2,3},{-2,-3}};
      Xt_int local_start[2] = {1,2};

      // create index section

      idxsection = xt_idxsection_new(start, num_dimensions, global_size[i >> 2],
                                     local_size[i & 3], local_start);

      // testing

      Xt_int ref_indices[16][6] =
        {{8,9,10,14,15,16},
         {10,9,8,16,15,14},
         {14,15,16,8,9,10},
         {16,15,14,10,9,8},
         {9,8,7,15,14,13},
         {7,8,9,13,14,15},
         {15,14,13,9,8,7},
         {13,14,15,7,8,9},
         {20,21,22,14,15,16},
         {22,21,20,16,15,14},
         {14,15,16,20,21,22},
         {16,15,14,22,21,20},
         {21,20,19,15,14,13},
         {19,20,21,13,14,15},
         {15,14,13,21,20,19},
         {13,14,15,19,20,21}};

      check_idxlist(idxsection, ref_indices[i], 6);

      // clean up

      xt_idxlist_delete(idxsection);
    }
  }

  { // test intersection of 2D section with negative global size

    for (int i = 0; i < 16; ++i) {
      for (int j = 0; j < 16; ++j) {

        Xt_idxlist idxsection_a, idxsection_b;
        Xt_idxlist idxvec_a, idxvec_b;
        Xt_idxlist idxsection_intersection;
        Xt_idxlist idxsection_intersection_other;
        Xt_idxlist idxvec_intersection;

        Xt_int start = 0;
        unsigned num_dimensions = 2;
        Xt_int global_size[4][2] = {{5,10},{5,-10},{-5,10},{-5,-10}};
        Xt_int local_size [4][2] = {{3,4},{3,-4},{-3,4},{-3,-4}};
        Xt_int local_start[2] = {1,2};

        Xt_int indices[16][12] =
          {{12, 13, 14, 15, 22, 23, 24, 25, 32, 33, 34, 35},
           {15, 14, 13, 12, 25, 24, 23, 22, 35, 34, 33, 32},
           {32, 33, 34, 35, 22, 23, 24, 25, 12, 13, 14, 15},
           {35, 34, 33, 32, 25, 24, 23, 22, 15, 14, 13, 12},
           {17, 16, 15, 14, 27, 26, 25, 24, 37, 36, 35, 34},
           {14, 15, 16, 17, 24, 25, 26, 27, 34, 35, 36, 37},
           {37, 36, 35, 34, 27, 26, 25, 24, 17, 16, 15, 14},
           {34, 35, 36, 37, 24, 25, 26, 27, 14, 15, 16, 17},
           {32, 33, 34, 35, 22, 23, 24, 25, 12, 13, 14, 15},
           {35, 34, 33, 32, 25, 24, 23, 22, 15, 14, 13, 12},
           {12, 13, 14, 15, 22, 23, 24, 25, 32, 33, 34, 35},
           {15, 14, 13, 12, 25, 24, 23, 22, 35, 34, 33, 32},
           {37, 36, 35, 34, 27, 26, 25, 24, 17, 16, 15, 14},
           {34, 35, 36, 37, 24, 25, 26, 27, 14, 15, 16, 17},
           {17, 16, 15, 14, 27, 26, 25, 24, 37, 36, 35, 34},
           {14, 15, 16, 17, 24, 25, 26, 27, 34, 35, 36, 37}};

        // create index section

        idxsection_a = xt_idxsection_new(start, num_dimensions, global_size[i >> 2],
                                         local_size[i & 3], local_start);
        idxsection_b = xt_idxsection_new(start, num_dimensions, global_size[j >> 2],
                                         local_size[j & 3], local_start);

        // create reference index vectors

        idxvec_a = xt_idxvec_new(indices[i], 12);
        idxvec_b = xt_idxvec_new(indices[j], 12);

        // testing

        idxsection_intersection = xt_idxlist_get_intersection(idxsection_a, idxsection_b);
        idxsection_intersection_other = xt_idxlist_get_intersection(idxsection_a,
                                                                    idxvec_b);
        idxvec_intersection = xt_idxlist_get_intersection(idxvec_a, idxvec_b);

        check_idxlist(idxsection_intersection,
                      xt_idxlist_get_indices_const(idxvec_intersection),
                      xt_idxlist_get_num_indices(idxvec_intersection));
        check_idxlist(idxsection_intersection_other,
                      xt_idxlist_get_indices_const(idxvec_intersection),
                      xt_idxlist_get_num_indices(idxvec_intersection));

        // clean up

        xt_idxlist_delete(idxsection_a);
        xt_idxlist_delete(idxsection_b);
        xt_idxlist_delete(idxvec_a);
        xt_idxlist_delete(idxvec_b);
        xt_idxlist_delete(idxsection_intersection);
        xt_idxlist_delete(idxsection_intersection_other);
        xt_idxlist_delete(idxvec_intersection);
      }
    }
  }

  { // test 2D section with negative global size

    Xt_idxlist idxsection;

    Xt_int start = 0;
    unsigned num_dimensions = 2;
    Xt_int global_size[2] = {-5,6};
    Xt_int local_size [2] = {-2,-3};
    Xt_int local_start[2] = {1,2};

    // create index section

    idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    // testing

    Xt_int ref_indices[6] = {16,15,14,22,21,20};

    check_idxlist(idxsection, ref_indices, 6);

    // check get_positions_of_indices

    Xt_int indices[34] = {-1,0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,14,16,17,18,19,
                          20,20,21,22,23,24,25,26,27,28,29,30};
    int ref_positions[34] = {-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,-1,2,1,-1,0,-1,-1,-1,
                             5,-1,4,3,-1,-1,-1,-1,-1,-1,-1,-1};
    int positions[34];

    if (xt_idxlist_get_positions_of_indices(idxsection, indices, 34, positions, 1) != 28)
      PUT_ERR("error in xt_idxlist_get_positions_of_indices"
              " (wrong number of unmatched indices)\n");

    for (int i = 0; i < 34; ++i)
      if (ref_positions[i] != positions[i])
        PUT_ERR("error in xt_idxlist_get_positions_of_indices (wrong position)\n");

    // clean up

    xt_idxlist_delete(idxsection);
  }

  { // test 2D section with stride in x

    Xt_idxlist idxsection;

    Xt_int start = 0;
    unsigned num_dimensions = 3;
    Xt_int global_size[3] = {5,5,2};
    Xt_int local_size [3] = {3,4,1};
    Xt_int local_start[3] = {2,0,1};

    // create index section

    idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    // testing

    Xt_int ref_indices[12] = {21, 23, 25, 27, 31, 33, 35, 37, 41, 43, 45, 47};

    check_idxlist(idxsection, ref_indices, 12);

    // clean up

    xt_idxlist_delete(idxsection);
  }

  { // test 2D section with stride in x and y

    Xt_idxlist idxsection;

    Xt_int start = 0;
    unsigned num_dimensions = 4;
    Xt_int global_size[4] = {3,2,5,2};
    Xt_int local_size [4] = {3,1,4,1};
    Xt_int local_start[4] = {0,1,1,0};

    // create index section

    idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    // testing

    Xt_int ref_indices[12] = {12, 14, 16, 18, 32, 34, 36, 38, 52, 54, 56, 58};

    check_idxlist(idxsection, ref_indices, 12);

    // clean up

    xt_idxlist_delete(idxsection);
  }

  { // check get_bounding_box

    Xt_idxlist idxsection;

    Xt_int start = 0;
    unsigned num_dimensions = 3;
    Xt_int global_size[3] = {4,4,4};
    Xt_int local_size [3] = {0,0,0};
    Xt_int local_start[3] = {2,0,1};

    // create index section

    idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    unsigned ndim = 3;
    Xt_int global_size_bb[ndim];
    Xt_int global_start_index = 0;
    struct Xt_bounds bounds[ndim];

    for (unsigned i = 0; i < ndim; ++i)
      global_size_bb[i] = 4;

    xt_idxlist_get_bounding_box(idxsection, ndim, global_size_bb,
                                global_start_index, bounds);

    for (unsigned i = 0; i < ndim; ++i)
      if (bounds[i].size != 0)
        PUT_ERR("ERROR: xt_idxlist_get_bounding_box\n");

    xt_idxlist_delete(idxsection);
  }

  { // check get_bounding_box

    Xt_idxlist idxsection;

    Xt_int start = 1;
    unsigned num_dimensions = 3;
    Xt_int global_size[3] = {5,4,3};
    Xt_int local_size [3] = {2,2,2};
    Xt_int local_start[3] = {2,2,1};

    // create index section

    idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    unsigned ndim = 3;
    Xt_int global_size_bb[ndim];
    Xt_int global_start_index = 1;
    struct Xt_bounds bounds[ndim];

    global_size_bb[0] = 5;
    global_size_bb[1] = 4;
    global_size_bb[2] = 3;

    xt_idxlist_get_bounding_box(idxsection, ndim, global_size_bb,
                                global_start_index, bounds);

    Xt_int ref_start[3] = {2,2,1};

    for (unsigned i = 0; i < ndim; ++i)
      if (bounds[i].size != 2 || bounds[i].start != ref_start[i])
        PUT_ERR("ERROR: xt_idxlist_get_bounding_box\n");

    xt_idxlist_delete(idxsection);
  }

  { // check get_bounding_box

    Xt_idxlist idxsection;

    Xt_int start = 1;
    unsigned num_dimensions = 4;
    Xt_int global_size[4] = {5,2,2,3};
    Xt_int local_size [4] = {2,2,1,2};
    Xt_int local_start[4] = {2,0,1,1};

    // create index section

    idxsection = xt_idxsection_new(start, num_dimensions, global_size,
                                   local_size, local_start);

    unsigned ndim = 3;
    Xt_int global_size_bb[ndim];
    Xt_int global_start_index = 1;
    struct Xt_bounds bounds[ndim];

    global_size_bb[0] = 5;
    global_size_bb[1] = 4;
    global_size_bb[2] = 3;

    xt_idxlist_get_bounding_box(idxsection, ndim, global_size_bb,
                                global_start_index, bounds);

    Xt_int ref_start[3] = {2,1,1};
    Xt_int ref_size[3] = {2,3,2};

    for (unsigned i = 0; i < ndim; ++i)
      if (bounds[i].size != ref_size[i] || bounds[i].start != ref_start[i])
        PUT_ERR("ERROR: xt_idxlist_get_bounding_box\n");

    xt_idxlist_delete(idxsection);
  }

  MPI_Finalize();

  return TEST_EXIT_CODE;
}

static void
do_tests(Xt_idxlist idxlist, Xt_int * ref_indices, Xt_int num_indices,
         struct Xt_stripe * ref_stripes, Xt_int ref_num_stripes) {

  check_idxlist(idxlist, ref_indices, num_indices);

  struct Xt_stripe * stripes;
  Xt_int num_stripes;

  xt_idxlist_get_index_stripes(idxlist, &stripes, &num_stripes);

  check_stripes(stripes, num_stripes, ref_stripes, ref_num_stripes);

  free(stripes);

  { // test packing and unpacking

    size_t buffer_size = xt_idxlist_get_pack_size(idxlist, MPI_COMM_WORLD);

    // allocate send buffers

    void* send_buffer, * recv_buffer;

    send_buffer = xmalloc(buffer_size);
    recv_buffer = xmalloc(buffer_size);

    // pack the index vector

    int position = 0;
    assert(buffer_size <= INT_MAX);
    xt_idxlist_pack(idxlist, send_buffer, (int)buffer_size, &position,
                    MPI_COMM_WORLD);

    // send the buffer

    int rank;

    xt_mpi_call(MPI_Comm_rank(MPI_COMM_WORLD, &rank), MPI_COMM_WORLD);

    MPI_Request request;

    xt_mpi_call(MPI_Isend(send_buffer, (int)buffer_size, MPI_PACKED, rank, 0,
                              MPI_COMM_WORLD, &request), MPI_COMM_WORLD);
    xt_mpi_call(MPI_Request_free(&request), MPI_COMM_WORLD);

    // receive the buffer

    xt_mpi_call(MPI_Recv(recv_buffer, (int)buffer_size, MPI_PACKED, rank, 0,
                         MPI_COMM_WORLD, MPI_STATUS_IGNORE), MPI_COMM_WORLD);

    // unpack the buffer

    Xt_idxlist idxlist_copy;

    position = 0;
    idxlist_copy
      = xt_idxlist_unpack(recv_buffer, (int)buffer_size, &position,
                          MPI_COMM_WORLD);

    // check copy

    check_idxlist(idxlist_copy, ref_indices, num_indices);

    // clean up

    free(send_buffer);
    free(recv_buffer);
    xt_idxlist_delete(idxlist_copy);
  }

  { // test copying

    Xt_idxlist idxlist_copy;

    idxlist_copy = xt_idxlist_copy(idxlist);

    // check copy

    check_idxlist(idxlist_copy, ref_indices, num_indices);

    // clean up

    xt_idxlist_delete(idxlist_copy);
  }
}

static void
check_stripes(struct Xt_stripe const * stripes, Xt_int num_stripes,
              struct Xt_stripe const * ref_stripes, Xt_int ref_num_stripes) {

  Xt_int i;

  if (num_stripes != ref_num_stripes)
    PUT_ERR("wrong number of stripes\n");

  for (i = 0; i < ref_num_stripes; ++i)
    if (stripes[i].start != ref_stripes[i].start ||
        stripes[i].nstrides != ref_stripes[i].nstrides ||
        stripes[i].stride != ref_stripes[i].stride)
      PUT_ERR("error in stripe (start, nstrides and/or stride)\n");
}
